'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/* --------------------------------------------------------------
 cart_handler.js 2021-12-31
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2021 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component for handling the add to cart and wishlist features
 * at the product details and the category listing pages. It cares
 * for attributes, properties, quantity and all other
 * relevant data for adding an item to the basket or wishlist
 */
gambio.widgets.module('cart_handler', ['hooks', 'form', 'xhr', 'loading_spinner', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

	'use strict';

	// ########## VARIABLE INITIALIZATION ##########

	var $this = $(this),
	    $body = $('body'),
	    $window = $(window),
	    busy = false,
	    ajax = null,
	    timeout = 0,
	    previousModifiers = {},
	    defaults = {
		// AJAX "add to cart" URL
		addCartUrl: 'shop.php?do=Cart/BuyProduct',
		// AJAX "add to cart" URL for customizer products
		addCartCustomizerUrl: 'shop.php?do=Cart/Add',
		// AJAX URL to perform a value check
		checkUrl: 'shop.php?do=CheckStatus',
		// AJAX URL to perform the add to wishlist
		wishlistUrl: 'shop.php?do=WishList/Add',
		// Submit URL for price offer button
		priceOfferUrl: 'gm_price_offer.php',
		// Submit method for price offer
		priceOfferMethod: 'get',
		// Selector for the cart dropdown
		dropdown: '#head_shopping_cart',
		// "Add to cart" buttons selectors
		cartButtons: '.js-btn-add-to-cart',
		// "Wishlist" buttons selectors
		wishlistButtons: '.btn-wishlist',
		// "Price offer" buttons selectors
		priceOfferButtons: '.btn-price-offer',
		// Selector for the attribute fields
		attributes: '.js-calculate',
		// Selector for product property
		productOptions: '.modifier-group .modifier-content .modifier-item',
		productOptionField: '.hidden-input',
		// Selector for the quantity
		quantity: '.js-calculate-qty',
		// URL where to get the theme for the dropdown
		tpl: null,
		// Show attribute images in product images swiper (if possible)
		// -- this feature is not supported yet --
		attributImagesSwiper: false,
		// Trigger the attribute images to this selectors
		triggerAttrImagesTo: '#product_image_swiper, #product_thumbnail_swiper, ' + '#product_thumbnail_swiper_mobile',
		// Class that gets added to the button on processing
		processingClass: 'loading',
		// Duration for that the success or fail class gets added to the button
		processingDuration: 2000,
		// AJAX response content selectors
		selectorMapping: {
			buttons: '.shopping-cart-button',
			giftContent: '.gift-cart-content-wrapper',
			giftLayer: '.gift-cart-layer',
			shareContent: '.share-cart-content-wrapper',
			shareLayer: '.share-cart-layer',
			hiddenOptions: '#cart_quantity .hidden-options',
			message: '.global-error-messages',
			messageCart: '.cart-error-msg',
			messageHelp: '.help-block',
			modelNumber: '.model-number',
			modelNumberText: '.model-number-text',
			price: '.current-price-container',
			modifiersForm: '.modifiers-selection',
			quantity: '.products-quantity-value',
			quantityInfo: '.products-quantity',
			ribbonSpecial: '.ribbon-special',
			shippingInformation: '#shipping-information-layer',
			shippingTime: '.products-shipping-time-value',
			shippingTimeImage: '.img-shipping-time img',
			totals: '#cart_quantity .total-box',
			weight: '.products-details-weight-container span',
			abroadShippingInfo: '.abroad-shipping-info'
		},
		page: 'product-listing'
	},
	    options = $.extend(true, {}, defaults, data),
	    module = {},
	    mobile = $(window).width() <= 767;

	// ########## HELPER FUNCTIONS ##########

	/**
  * Helper function that updates the button
  * state with an error or success class for
  * a specified duration
  * @param   {object}        $target         jQuery selection of the target button
  * @param   {string}        state           The state string that gets added to the loading class
  * @private
  */
	var _addButtonState = function _addButtonState($target, state) {
		var timer = setTimeout(function () {
			$target.removeClass(options.processingClass + ' ' + options.processingClass + state);
		}, options.processingDuration);

		$target.data('timer', timer).addClass(options.processingClass + state);
	};

	/**
  * Helper function to set the messages and the
  * button state.
  * @param       {object}    data                Result form the ajax request
  * @param       {object}    $form               jQuery selecion of the form
  * @param       {boolean}   disableButtons      If true, the button state gets set to (in)active
  * @param       {boolean}   showNoCombiMesssage If true, the error message for missing property combination selection will be displayed
  * @private
  */
	var _stateManager = function _stateManager(data, $form, disableButtons, showNoCombiSelectedMesssage) {

		// Remove the attribute images from the common content
		// so that it doesn't get rendered anymore. Then trigger
		// an event to the given selectors and deliver the
		// attrImages object
		if (options.attributImagesSwiper && data.attrImages && data.attrImages.length) {
			delete data.content.images;
			$(options.triggerAttrImagesTo).trigger(jse.libs.theme.events.SLIDES_UPDATE(), { attributes: data.attrImages });
		}

		// Set the messages given inside the data.content object
		$.each(data.content, function (i, v) {
			var $element = $body.hasClass('page-product-info') ? $this.find(options.selectorMapping[v.selector]) : $form.parent().find(options.selectorMapping[v.selector]);

			if ((!showNoCombiSelectedMesssage || v.value === '') && i === 'messageNoCombiSelected') {
				return true;
			}

			switch (v.type) {
				case 'hide':
					if (v.value === 'true') {
						$element.hide();
					} else {
						$element.show();
					}
					break;
				case 'html':
					$element.html(v.value);
					break;
				case 'attribute':
					$element.attr(v.key, v.value);
					break;
				case 'replace':
					if (v.value) {
						$element.replaceWith(v.value);
					} else {
						$element.addClass('hidden').empty();
					}
					break;
				default:
					$element.text(v.value);
					break;
			}
		});

		// Dis- / Enable the buttons
		if (disableButtons) {
			var $buttons = $form.find(options.cartButtons);
			if (data.success) {
				$buttons.removeClass('inactive');
				$buttons.removeClass('btn-inactive');
				$buttons.prop("disabled", false);
			} else {
				$buttons.addClass('inactive');
				$buttons.addClass('btn-inactive');
				$buttons.prop("disabled", true);
			}
		}

		if (data.content.message) {
			var $errorField = $form.find(options.selectorMapping[data.content.message.selector]);
			if (data.content.message.value) {
				$errorField.removeClass('hidden').show();
			} else {
				$errorField.addClass('hidden').hide();

				if (showNoCombiSelectedMesssage && data.content.messageNoCombiSelected !== undefined && data.content.messageNoCombiSelected) {
					if (data.content.messageNoCombiSelected.value) {
						$errorField.removeClass('hidden').show();
					} else {
						$errorField.addClass('hidden').hide();
					}
				}
			}
		}

		$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
	};

	/**
  * Helper function to send the ajax
  * On success redirect to a given url, open a layer with
  * a message or add the item to the cart-dropdown directly
  * (by triggering an event to the body)
  * @param       {object}      data      Form data
  * @param       {object}      $form     The form to fill
  * @param       {string}      url       The URL for the AJAX request
  * @private
  */
	var _addToSomewhere = function _addToSomewhere(data, $form, url, $button) {
		function callback() {
			jse.libs.xhr.post({ url: url, data: data }, true).done(function (result) {
				try {
					// Fill the page with the result from the ajax
					_stateManager(result, $form, false);

					// If the AJAX was successful execute
					// a custom functionality
					if (result.success) {
						switch (result.type) {
							case 'url':
								if (result.url.substr(0, 4) !== 'http') {
									location.href = jse.core.config.get('appUrl') + '/' + result.url;
								} else {
									location.href = result.url;
								}

								break;
							case 'dropdown':
								$body.trigger(jse.libs.theme.events.CART_UPDATE(), [true]);
								break;
							case 'layer':
								jse.libs.theme.modal.info({ title: result.title, content: result.msg });
								break;
							default:
								break;
						}
					}
				} catch (ignore) {}
				_addButtonState($button, '-success');
			}).fail(function () {
				_addButtonState($button, '-fail');
			}).always(function () {
				// Reset the busy flag to be able to perform
				// further AJAX requests
				busy = false;
			});
		}

		if (!busy) {
			// only execute the ajax
			// if there is no pending ajax call
			busy = true;

			jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.add, data, 500).then(callback).catch(callback);
		}
	};

	// ########## EVENT HANDLER ##########

	/**
  * Handler for the submit form / click
  * on "add to cart" & "wishlist" button.
  * It performs a check on the availability
  * of the combination and quantity. If
  * successful it performs the add to cart
  * or wishlist action, if it's not a
  * "check" call
  * @param       {object}    e      jQuery event object
  * @private
  */
	var _submitHandler = function _submitHandler(e) {
		if (e) {
			e.preventDefault();
		}

		var $self = $(this),
		    $form = $self.is('form') ? $self : $self.closest('form'),
		    customizer = $form.hasClass('customizer'),
		    properties = !!$form.find('.properties-selection-form').length,
		    module = properties ? '' : '/Attributes',
		    showNoCombiSelectedMesssage = e && e.data && e.data.target && e.data.target !== 'check';

		if ($form.length) {

			// Show properties overlay
			// to disable user interaction
			// before markup replace
			if (properties) {
				$this.addClass('loading');
			}

			if ($self.is('select')) {
				var price = $self.find(":selected").attr('data-price');
				$self.parents('.modifier-group').find('.selected-value-price').text(price);
			}

			var getGalleryHash = $('#current-gallery-hash').val();
			$form.find('#update-gallery-hash').val(getGalleryHash);

			var formdata = jse.libs.form.getData($form, null, true);
			formdata.target = e && e.data && e.data.target ? e.data.target : 'check';
			formdata.isProductInfo = $form.hasClass('product-info') ? 1 : 0;

			// Abort previous check ajax if
			// there is one in progress
			if (ajax && e) {
				ajax.abort();
			}

			// Add processing-class to the button
			// and remove old timed events
			if (formdata.target !== 'check') {
				var timer = $self.data('timer');
				if (timer) {
					clearTimeout(timer);
				}

				$self.removeClass(options.processingClass + '-success ' + options.processingClass + '-fail').addClass(options.processingClass);
			}

			formdata.previousModifiers = previousModifiers;

			ajax = jse.libs.xhr.get({
				url: options.checkUrl + module,
				data: formdata
			}, true).done(function (result) {
				_stateManager(result, $form, true, showNoCombiSelectedMesssage);
				$this.removeClass('loading');

				if (result.success) {
					var event = null,
					    url = null;

					switch (formdata.target) {
						case 'wishlist':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_WISHLIST();
							}
							url = options.wishlistUrl;
							break;
						case 'cart':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_CART();
								url = options.addCartCustomizerUrl;
							} else {
								url = options.addCartUrl;
							}
							break;
						case 'price_offer':
							$form.attr('action', options.priceOfferUrl).attr('method', options.priceOfferMethod);
							$form.off('submit');
							$form.submit();

							return;
						default:
							setTimeout(function () {
								$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
							}, 250);
							break;
					}

					if (event) {
						var deferred = $.Deferred();
						deferred.done(function (customizerRandom) {
							formdata[customizerRandom] = 0;
							_addToSomewhere(formdata, $form, url, $self);
						}).fail(function () {
							_addButtonState($self, '-fail');
						});
						$body.trigger(event, [{ 'deferred': deferred, 'dataset': formdata }]);
					} else if (url) {
						_addToSomewhere(formdata, $form, url, $self);
					}

					// Check if the gallery images changed
					if (formdata.target === 'check' && result.content.imageGallery.trim() !== '' && result.content.replaceGallery === true && formdata.isProductInfo === 1) {
						var loadingSpinner = jse.libs.loading_spinner.show($('.product-info-stage'), 9999);

						var swipers = [$('#product_image_swiper'), $('#product_thumbnail_swiper'), $('#product_thumbnail_swiper_mobile')];

						var _iteratorNormalCompletion = true;
						var _didIteratorError = false;
						var _iteratorError = undefined;

						try {
							for (var _iterator = swipers[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
								var element = _step.value;

								var instance = element.swiper();

								if ((typeof instance === 'undefined' ? 'undefined' : _typeof(instance)) !== 'object') {
									continue;
								}

								instance.destroy(true, true);
								element.off().remove();
							}
						} catch (err) {
							_didIteratorError = true;
							_iteratorError = err;
						} finally {
							try {
								if (!_iteratorNormalCompletion && _iterator.return) {
									_iterator.return();
								}
							} finally {
								if (_didIteratorError) {
									throw _iteratorError;
								}
							}
						}

						$('#image-collection-container').html(result.content.imageGallery);
						$('#product_image_layer').html(result.content.imageModal);

						gambio.widgets.init($('.product-info-content')).done(function () {
							jse.libs.loading_spinner.hide(loadingSpinner);
						});
					} else if (formdata.target === 'check' && result.content.imageGallery.trim() === '' && result.content.replaceGallery === true) {
						$('#image-collection-container').html(result.content.imageGallery);
						$('#product_image_layer').html(result.content.imageModal);
					}

					previousModifiers = formdata.modifiers;
				}
			}).fail(function () {
				_addButtonState($self, '-fail');
			});
		}
	};

	/**
  * Handler for the change property option
  * */
	var _changeProductOptions = function _changeProductOptions(e) {
		var option = e.currentTarget;
		var optionValue = $(option).data('value');
		var optionContainer = $(option).parents('.modifier-group');

		$(optionContainer).find('li.active').removeClass('active');
		$(optionContainer).find('.modifier-item.active-modifier').removeClass('active-modifier');
		$(optionContainer).find('input.hidden-input').val(optionValue);
		$(optionContainer).find('input.hidden-input').trigger('blur', []);

		$(option).parents('li').addClass('active');
		$(option).addClass('active-modifier');
	};

	var _selectSelectedModifierInfo = function _selectSelectedModifierInfo(e) {
		var option = e.currentTarget;
		var price = $(option).attr('data-price');
		var label = $(option).attr('data-label');
		$(option).parents('.modifier-group').find('.selected-value-price').removeClass('temporary-value').attr('data-default-price', price);
		$(option).parents('.modifier-group').find('.selected-value').attr('data-default-value', label);
	};

	var _setSelectedModifierInfo = function _setSelectedModifierInfo(e) {
		var option = e.currentTarget;
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var price = $(option).attr('data-price');
			var label = $(option).attr('data-label');
			$(option).parents('.modifier-group').find('.selected-value-price').addClass('temporary-value').text(price);
			$(option).parents('.modifier-group').find('.selected-value').text(label);
		}
	};

	var _resetSelectedModifierInfo = function _resetSelectedModifierInfo(e) {
		var option = $(this);
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var priceHolder = $(option).parents('.modifier-group').find('.selected-value-price');
			var labelHolder = $(option).parents('.modifier-group').find('.selected-value');
			$(priceHolder).removeClass('temporary-value').text($(priceHolder).attr('data-default-price'));
			$(labelHolder).text($(labelHolder).attr('data-default-value'));
		}
	};

	/**
  * Keyup handler for quantity input field
  *
  * @param e
  * @private
  */
	var _keyupHandler = function _keyupHandler(e) {
		clearTimeout(timeout);

		timeout = setTimeout(function () {
			_submitHandler.call(this, e);
		}.bind(this), 300);
	};

	/**
  * Event handler for the add to cart button, that shows or hides the throbber.
  */
	var _addToCartThrobberHandler = function _addToCartThrobberHandler(e) {
		var $btn = $(this);
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var formReady = true;

		$(".properties-selection-form select").each(function () {
			var val = $(this).val();
			if (!val || val < 1) {
				formReady = false;
			}
		});

		if (formReady) {
			$btn.hide();
			$btnFake.show().prop("disabled", true).prepend('<span class="throbbler"></span>');
		}
	};

	/**
  * Cart dropdown oben event handler for the body.
  */
	var _cartDropdownOpenHandler = function _cartDropdownOpenHandler(e) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var fakeOrigLabel = $btnFake.html();
		var productCount = $(".cart-products-count").html();

		var textPhrases = JSON.parse($('#product-details-text-phrases').html());
		console.log(textPhrases['productsInCartSuffix']);

		$btnFake.html("<i class=\"fa fa-check\"></i> " + parseInt(productCount) + textPhrases['productsInCartSuffix']).prop("disabled", true).addClass("btn-buy-complete");

		setTimeout(function () {
			$btnFake.html(fakeOrigLabel).removeClass("btn-buy-complete").hide().prop("disabled", false);
			$(".throbbler", $btn).remove();
			$btn.show();
		}, 5000);
	};

	// ########## INITIALIZATION ##########

	/**
  * Init function of the widget
  * @constructor
  */
	module.init = function (done) {

		var $forms = $this.find('form');

		if (options.page === 'product-info') {
			$forms.find("[name=btn-add-to-cart]").on('touchstart touchmove touchend touchcancel', function () {
				return $forms.find("[name=btn-add-to-cart]").click();
			});
			$forms.find("[name=btn-add-to-cart]").on('mouseup', _addToCartThrobberHandler);
			$("body").on('CART_DROPDOWN_OPEN', _cartDropdownOpenHandler);
		}

		$forms.on('submit', { 'target': 'cart' }, _submitHandler).on('click', options.wishlistButtons, { 'target': 'wishlist' }, _submitHandler).on('click', options.priceOfferButtons, { 'target': 'price_offer' }, _submitHandler).on('change', options.attributes, { 'target': 'check' }, _submitHandler).on('mouseover', options.attributes, _setSelectedModifierInfo).on('mouseout', options.attributes, _resetSelectedModifierInfo).on('blur', options.productOptionField, { 'target': 'check' }, _submitHandler).on('click', options.productOptions, { 'target': 'check' }, function (e) {
			_selectSelectedModifierInfo(e);
			_changeProductOptions(e);
		}).on('mouseover', options.productOptions, _setSelectedModifierInfo).on('mouseout', options.productOptions, _resetSelectedModifierInfo).on('blur', options.quantity, { 'target': 'check' }, function (e) {
			_submitHandler(e);
		}).on('keyup', options.quantity, { 'target': 'check' }, _keyupHandler);

		// Fallback if the backend renders incorrect data
		// on initial page call
		$forms.not('.no-status-check').not('.product-info').each(function () {
			_submitHandler.call($(this));
		});
		done();
	};

	// Return data to widget engine
	return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
